﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

namespace CS_sample
{
    #region CRW関数呼出し全クラスの基底クラス
    /// <summary>
    /// CRW関数呼出し全クラスの基底クラス
    /// </summary>
    abstract class CrwFunction
    {
        #region メンバフィールド定義
        public delegate void CrwFunctionResultHandler(CrwFunction crwFunction); // 関数宣言
        protected event CrwFunctionResultHandler CrwFunctionResult;             // CRW関数の実行結果を表示する関数
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="crwFunctionResultHandler"></param>
        protected CrwFunction(CrwFunctionResultHandler crwFunctionResultHandler)
        {
            CrwFunctionResult += new CrwFunctionResultHandler(crwFunctionResultHandler);
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        public void Call()
        {
            DoCall();
            CrwFunctionResult(this);
        }

        /// <summary>
        /// 各CRW関数の呼出し処理


        /// </summary>
        protected abstract void DoCall();
    }
    #endregion

    #region (1) シリアルポート名取得関数呼出しクラス
    /// <summary>
    /// (1) シリアルポート名取得関数呼出しクラス
    /// </summary>
    class CrwGetPortNamesFunction : CrwFunction
    {
        #region シリアルポート名取得関数 crwGetPortNames()
        /// <summary>
        /// シリアルポート名取得関数 crwGetPortNames()
        /// </summary>
        /// <param name="bpPortNames">使用可能なシリアルポート名をCSV形式で返却するポインタ</param>
        /// <param name="piNumOfPort">使用可能なシリアルポートの数を返却するポインタ</param>
        /// <returns>ブール型, 論理型(Boolean)<</returns>
        [DllImport("PCardRW32.dll")]
        static extern bool crwGetPortNames(
                                         ref System.IntPtr bpPortNames,
                                         ref short piNumOfPort);
        #endregion

        #region メンバフィールド定義
        public bool _ret;           // 
        public System.IntPtr _bpPortNames; // 
        public short _piNumOfPort;  // 
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="bpPortNames"></param>
        /// <param name="piNumOfPort"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwGetPortNamesFunction(
                                        ref System.IntPtr bpPortNames,
                                        ref short piNumOfPort,
                                        CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            // baseはコンストラクタを呼び出し、引数を加えるためのものかな

            _ret = false;
            _bpPortNames = bpPortNames;
            _piNumOfPort = piNumOfPort;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwGetPortNames(
                                         ref _bpPortNames,
                                         ref _piNumOfPort);
        }
    }
    #endregion

    #region (2) COMポート情報取得関数呼出しクラス
    /// <summary>
    /// (2) COMポート情報取得関数呼出しクラス
    /// </summary>
    class CrwGetComDevInfoFunction : CrwFunction
    {
        #region ＣＯＭポート情報取得関数 crwGetComDevInfo()
        /// <summary>
        /// ＣＯＭポート情報取得関数 crwGetComDevInfo()
        /// </summary>
        /// <param name="iPortNo">取得したいシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="bpDriveDesc">シリアルポートのディスクリプター名を返却するエリアのポインタ</param>
        /// <param name="bpProviderName">シリアルポートドライバーのプロバイダーを返却するエリアのポインタ</param>
        /// <param name="bpDriverDate">シリアルポートドライバーの日付を返却するエリアのポインタ</param>
        /// <param name="bpDriverVersion">シリアルポートドライバーのバージョンを返却するエリアのポインタ</param>
        /// <returns>ブール型, 論理型(Boolean)</returns>        
        [DllImport("PCardRW32.dll")]
        static extern bool crwGetComDevInfo(
                                         short iPortNo,
                                         ref System.IntPtr bpDriveDesc,
                                         ref System.IntPtr bpProviderName,
                                         ref System.IntPtr bpDriverDate,
                                         ref System.IntPtr bpDriverVersion);
        #endregion

        #region メンバフィールド定義
        public bool _ret;
        public short _iPortNo;
        public System.IntPtr _bpDriveDesc;
        public System.IntPtr _bpProviderName;
        public System.IntPtr _bpDriverDate;
        public System.IntPtr _bpDriverVersion;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="bpDriveDesc"></param>
        /// <param name="bpProviderName"></param>
        /// <param name="bpDriverDate"></param>
        /// <param name="bpDriverVersion"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwGetComDevInfoFunction(short iPortNo,
                                         ref System.IntPtr bpDriveDesc,
                                         ref System.IntPtr bpProviderName,
                                         ref System.IntPtr bpDriverDate,
                                         ref System.IntPtr bpDriverVersion,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = false;
            _iPortNo = iPortNo;
            _bpDriveDesc = bpDriveDesc;
            _bpProviderName = bpProviderName;
            _bpDriverDate = bpDriverDate;
            _bpDriverVersion = bpDriverVersion;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwGetComDevInfo(
                                         _iPortNo,
                                         ref _bpDriveDesc,
                                         ref _bpProviderName,
                                         ref _bpDriverDate,
                                         ref _bpDriverVersion);
        }
    }
    #endregion

    #region (3) ポートオープン関数呼出しクラス
    /// <summary>
    /// (3) ポートオープン関数呼出しクラス
    /// </summary>
    class CrwOpenPortFunction : CrwFunction
    {
        #region ポートオープン関数 crwOpenPort()
        /// <summary>
        /// ポートオープン関数 crwOpenPort()
        /// </summary>
        /// <param name="iPortNo">オープンするシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="lBaudRate">シリアルポートの通信速度（2400、4800、9600、19200、38400、57600、115200）</param>
        /// <param name="lpDataBit">シリアルポートのデータビット長（7または8）</param>
        /// <param name="piNumOfPort">シリアルポートのパリティチェック（ N：なし ／ E：偶数 ／ O：奇数）</param>
        /// <returns>整数型(short)</returns>   
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwOpenPort(
                                         short iPortNo,
                                         uint lBaudRate,
                                         char* lpDataBit,
                                         char* lpParity);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lBaudRate;
        public /*unsafe char**/char _lpDataBit;
        public /*unsafe char**/char _lpParity;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lBaudRate"></param>
        /// <param name="lpDataBit"></param>
        /// <param name="lpParity"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwOpenPortFunction(
                                         short iPortNo,
                                         uint lBaudRate,
                                         char* lpDataBit,
                                         char* lpParity,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lBaudRate = lBaudRate;
            _lpDataBit = /*lpDataBit*/*lpDataBit;
            _lpParity = /*lpParity*/*lpParity;
        }

        /// <summary>
        /// CRW関数呼出し
        /// </summary>
        protected unsafe override void DoCall()
        {
            char lpDataBit = _lpDataBit;
            char lpParity = _lpParity;
            _ret = crwOpenPort(
                                         _iPortNo,
                                         _lBaudRate,
                                        /*_lpDataBit*/&lpDataBit,
                                        /*_lpParity*/&lpParity);
        }
    }
    #endregion

    #region (4) ポートクローズ関数呼出しクラス
    /// <summary>
    /// (4) ポートクローズ関数呼出しクラス
    /// </summary>
    class CrwClosePortFunction : CrwFunction
    {
        #region ポートクローズ関数 crwClosePort()
        /// <summary>
        /// ポートクローズ関数 crwClosePort()
        /// </summary>
        /// <param name="iPortNo">クローズするシリアルポートの番号（例：COM1なら”1”）</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwClosePort(short iPortNo);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwClosePortFunction(
                                         short iPortNo,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwClosePort(_iPortNo);
        }
    }
    #endregion

    #region (5) コマンド送受信関数呼出しクラス
    /// <summary>
    /// (5) コマンド送受信関数呼出しクラス
    /// </summary>
    class CrwSendCommandRRFunction : CrwFunction
    {
        #region コマンド送受信関数 crwSendCommandRR()
        /// <summary>
        /// コマンド送受信関数 crwSendCommandRR() 
        /// </summary>
        /// <param name="iPortNo">送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iRecMode">ビジーステータス受信の時にポーリング再送を行なうかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lTimeOut">ACK/NAK待ち及びCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="lBusyTimeOut">ビジーステータスの終了待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ　（0:しない／0以外:する）</param>
        /// <param name="lpCommandCode">機器のコマンドコード（2バイト）</param>
        /// <param name="lpCommandParameter">機器のコマンドのパラメータ(パラメータ無しの時はNull)</param>
        /// <param name="iParameterLength">コマンドパラメータの長さ</param>
        /// <param name="bpCOMSTS">レスポンスのコマンドコード(=COM$)・レディステータスのステータス情報(=STS$)を返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpERRSEN">レスポンスのエラーコード(=ERR$)・レディステータスのセンサー情報(=SEN$)を返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分(=DATA$)を返却するエリアのポインタ(センサー情報以外の追加情報がある場合は、以降の情報は“bpResDATA”に格納される)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwSendCommandRR(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         uint lBusyTimeOut,
                                         short iDSRCheck,
                                         string lpCommandCode,
                                         string lpCommandParameter,
                                         short iParameterLength,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN,
                                         ref System.IntPtr bpResDATA);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iRecMode;
        public uint _lTimeOut;
        public uint _lBusyTimeOut;
        public short _iDSRCheck;
        public string _lpCommandCode;
        public string _lpCommandParameter;
        public short _iParameterLength;
        public System.IntPtr _bpCOMSTS;
        public System.IntPtr _bpERRSEN;
        public System.IntPtr _bpResDATA;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRecMode"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="lBusyTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpCommandCode"></param>
        /// <param name="lpCommandParameter"></param>
        /// <param name="iParameterLength"></param>
        /// <param name="bpCOMSTS"></param>
        /// <param name="bpERRSEN"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendCommandRRFunction(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         uint lBusyTimeOut,
                                         short iDSRCheck,
                                         string lpCommandCode,
                                         string lpCommandParameter,
                                         short iParameterLength,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN,
                                         ref System.IntPtr bpResDATA,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRecMode = iRecMode;
            _lTimeOut = lTimeOut;
            _lBusyTimeOut = lBusyTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpCommandCode = lpCommandCode;
            _lpCommandParameter = lpCommandParameter;
            _iParameterLength = iParameterLength;
            _bpCOMSTS = bpCOMSTS;
            _bpERRSEN = bpERRSEN;
            _bpResDATA = bpResDATA;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendCommandRR(
                                         _iPortNo,
                                         _iRecMode,
                                         _lTimeOut,
                                         _lBusyTimeOut,
                                         _iDSRCheck,
                                         _lpCommandCode,
                                         _lpCommandParameter,
                                         _iParameterLength,
                                         ref _bpCOMSTS,
                                         ref _bpERRSEN,
                                         ref _bpResDATA);
        }
    }
    #endregion

    #region (6) コマンド送信関数呼出しクラス
    /// <summary>
    /// (6) コマンド送信関数呼出しクラス
    /// </summary>
    class CrwSendCommandFunction : CrwFunction
    {
        #region コマンド送信関数 crwSendCommand()
        /// <summary>
        /// コマンド送信関数 crwSendCommand()
        /// </summary>
        /// <param name="iPortNo">コマンドを送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="lTimeOut">ACK/NAK待ち及びCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lpCommandCode">機器のコマンドコード（2バイト）</param>
        /// <param name="lpCommandParameter">機器のコマンドのパラメータ（パラメータ無しの時はNULL）</param>
        /// <param name="iParameterLength">コマンドパラメータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwSendCommand(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpCommandCode,
                                         string lpCommandParameter,
                                         short iParameterLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public string _lpCommandCode;
        public string _lpCommandParameter;
        public short _iParameterLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpCommandCode"></param>
        /// <param name="lpCommandParameter"></param>
        /// <param name="iParameterLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendCommandFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpCommandCode,
                                         string lpCommandParameter,
                                         short iParameterLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpCommandCode = lpCommandCode;
            _lpCommandParameter = lpCommandParameter;
            _iParameterLength = iParameterLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendCommand(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         _lpCommandCode,
                                         _lpCommandParameter,
                                         _iParameterLength);
        }
    }
    #endregion

    #region (7) レスポンス受信関数呼出しクラス
    /// <summary>
    /// (7) レスポンス受信関数呼出しクラス
    /// </summary>
    class CrwReceiveResponseFunction : CrwFunction
    {
        #region レスポンス受信関数  crwReceiveResponse()
        /// <summary>
        /// レスポンス受信関数  crwReceiveResponse()
        /// </summary>
        /// <param name="iPortNo">レスポンスを受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iRecMode">ビジーステータス受信時にポーリング再送を行うかどうかのフラグ</param>
        /// <param name="lTimeOut">ポーリングに対する機器からの応答待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線のチェックを行うかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="bpCOMSTS">ステータス情報、またはコマンドコードを返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpERRSEN">センサー情報、またはエラーコードを返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分(=DATA$)を返却するエリアのポインタ(センサー情報以外の追加情報がある場合は、以降の情報は“bpResDATA”に格納される)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwReceiveResponse(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN,
                                         ref System.IntPtr bpResDATA);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iRecMode;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public System.IntPtr _bpCOMSTS;
        public System.IntPtr _bpERRSEN;
        public System.IntPtr _bpResDATA;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRecMode"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="bpCOMSTS"></param>
        /// <param name="bpERRSEN"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwReceiveResponseFunction(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN,
                                         ref System.IntPtr bpResDATA,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRecMode = iRecMode;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _bpCOMSTS = bpCOMSTS;
            _bpERRSEN = bpERRSEN;
            _bpResDATA = bpResDATA;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwReceiveResponse(
                                         _iPortNo,
                                         _iRecMode,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         ref _bpCOMSTS,
                                         ref _bpERRSEN,
                                         ref _bpResDATA);
        }
    }
    #endregion

    #region (8) データ送信関数(応答あり)呼出しクラス
    /// <summary>
    /// (8) データ送信関数(応答あり)呼出しクラス
    /// </summary>
    class CrwSendDataWRFunction : CrwFunction
    {
        #region データ送信関数（応答あり）crwSendDataWR()
        /// <summary>
        /// データ送信関数（応答あり）crwSendDataWR()
        /// </summary>
        /// <param name="iPortNo">データを送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="lTimeOut">ACK/NAK待ち及びCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwSendDataWR(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpData,
                                         short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public string _lpData;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendDataWRFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpData,
                                         short iDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpData = lpData;
            _iDataLength = iDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendDataWR(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         _lpData,
                                         _iDataLength);
        }
    }
    #endregion

    #region (9) データ送信関数(応答なし)呼出しクラス
    /// <summary>
    /// (9) データ送信関数(応答なし)呼出しクラス
    /// </summary>
    class CrwSendDataQRFunction : CrwFunction
    {
        #region データ送信関数（応答なし） crwSendDataQR()
        /// <summary>
        /// データ送信関数（応答なし） crwSendDataQR()
        /// </summary>
        /// <param name="iPortNo">データを送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="lTimeOut">ACK/NAK待ち及びCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwSendDataQR(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpData,
                                         short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public string _lpData;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendDataQRFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         string lpData,
                                         short iDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpData = lpData;
            _iDataLength = iDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendDataQR(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         _lpData,
                                         _iDataLength);
        }
    }
    #endregion

    #region (10) データ受信関数呼出しクラス
    /// <summary>
    /// (10) データ受信関数呼出しクラス
    /// </summary>
    class CrwReceiveDataFunction : CrwFunction
    {
        #region データ受信関数 crwReceiveData()
        /// <summary>
        /// データ受信関数 crwReceiveData()
        /// </summary>
        /// <param name="iPortNo">受信するシリアルポートの番号</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外:する / 0:しない）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分を返却するエリアのポインタ</param>
        /// <param name="iDataLength">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwReceiveData(
                                         short iPortNo,
                                         short iDSRCheck,
                                         ref System.IntPtr bpResDATA,
                                         ref short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iDSRCheck;
        public System.IntPtr _bpResDATA;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwReceiveDataFunction(
                                         short iPortNo,
                                         short iDSRCheck,
                                         ref System.IntPtr bpResDATA,
                                         ref short iDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iDSRCheck = iDSRCheck;
            _bpResDATA = bpResDATA;
            _iDataLength = iDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwReceiveData(
                                         _iPortNo,
                                         _iDSRCheck,
                                         ref _bpResDATA,
                                         ref _iDataLength);
        }
    }
    #endregion

    #region (11) 電文データ送受信関数呼出しクラス
    /// <summary>
    /// (11) 電文データ送受信関数呼出しクラス
    /// </summary>
    class CrwSendDataRRFunction : CrwFunction
    {
        #region 電文データ送受信関数 crwSendDataRR()
        /// <summary>
        /// 電文データ送受信関数 crwSendDataRR()
        /// </summary>
        /// <param name="iPortNo">送受信するシリアルポート番号</param>
        /// <param name="iRetryCount">データ送信・応答データ受信タイムアウト時のリトライ回数</param>
        /// <param name="lTimeOut">リトライ時の待ち時間（単位:msec）</param>
        /// <param name="iDSRCheck">信号線をチェックするかどうかのフラグ（0以外:する / 0:しない）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <param name="bpResDATA">レスポンスのデータ部分を返却するエリアポインタ</param>
        /// <param name="piDataLength">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSendDataRR(
                                                    short iPortNo,
                                                    short iRetryCount,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    string lpData,
                                                    short iDataLength,
                                                    ref System.IntPtr bpResDATA,
                                                    ref short piDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iRetryCount;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public string _lpData;
        public short _iDataLength;
        public System.IntPtr _bpResDATA;
        public short _piDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="piDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendDataRRFunction(
                                                    short iPortNo,
                                                    short iRetryCount,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    string lpData,
                                                    short iDataLength,
                                                    ref System.IntPtr bpResDATA,
                                                    ref short piDataLength,
                                                    CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRetryCount = iRetryCount;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpData = lpData;
            _iDataLength = iDataLength;
            _bpResDATA = bpResDATA;
            _piDataLength = piDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendDataRR(
                                                    _iPortNo,
                                                    _iRetryCount,
                                                    _lTimeOut,
                                                    _iDSRCheck,
                                                    _lpData,
                                                    _iDataLength,
                                                    ref _bpResDATA,
                                                    ref _piDataLength);
        }
    }
    #endregion

    #region (12) 電文ﾃﾞｰﾀ送信関数呼出しクラス
    /// <summary>
    /// (12) 電文ﾃﾞｰﾀ送信関数呼出しクラス
    /// </summary>
    class CrwSendDataBCCFunction : CrwFunction
    {
        #region 電文データ送信関数 crwSendDataBCC()
        /// <summary>
        /// 電文データ送信関数 crwSendDataBCC()
        /// </summary>
        /// <param name="iPortNo">データを送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iACKCheck">ACK/NAKを待つかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="iRetryCount">データ送信・ACK/NAK受信タイムアウト時のリトライ回数</param>
        /// <param name="lTimeOut">ACK/NAK待ち及びCTS ON待ちタイマー（単位:msec）(IACKCheck=0のときはACK/NAK待ちタイマーのみ無効)</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSendDataBCC(
                                                    short iPortNo,
                                                    short iACKCheck,
                                                    short iRetryCount,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    string lpData,
                                                    short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iACKCheck;
        public short _iRetryCount;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public string _lpData;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iACKCheck"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSendDataBCCFunction(
                                                    short iPortNo,
                                                    short iACKCheck,
                                                    short iRetryCount,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    string lpData,
                                                    short iDataLength,
                                                    CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iACKCheck = iACKCheck;
            _iRetryCount = iRetryCount;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpData = lpData;
            _iDataLength = iDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSendDataBCC(
                                                    _iPortNo,
                                                    _iACKCheck,
                                                    _iRetryCount,
                                                    _lTimeOut,
                                                    _iDSRCheck,
                                                    _lpData,
                                                    _iDataLength);
        }
    }
    #endregion

    #region (13) 電文ﾃﾞｰﾀ受信関数呼出しクラス
    /// <summary>
    /// (13) 電文ﾃﾞｰﾀ受信関数呼出しクラス
    /// </summary>
    class CrwReceiveDataBCCFunction : CrwFunction
    {
        #region 電文データ受信関数 crwReceiveDataBCC()
        /// <summary>
        /// 電文データ受信関数 crwReceiveDataBCC()
        /// </summary>
        /// <param name="iPortNo">レスポンスを受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iACKResponse">ACK/NAKを返却するかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lTimeOut">CTS ON待ちタイマー（単位:msec）   / iACKResponse =0のときは無効</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外:する / 0:しない）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分を返却するエリアのポインタ</param>
        /// <param name="piDataLength">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwReceiveDataBCC(
                                                    short iPortNo,
                                                    short iACKResponse,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    ref System.IntPtr bpResDATA,
                                                    ref short piDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iACKResponse;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public System.IntPtr _bpResDATA;
        public short _piDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iACKResponse"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="piDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwReceiveDataBCCFunction(
                                                    short iPortNo,
                                                    short iACKResponse,
                                                    uint lTimeOut,
                                                    short iDSRCheck,
                                                    ref System.IntPtr bpResDATA,
                                                    ref short piDataLength,
                                                    CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iACKResponse = iACKResponse;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _bpResDATA = bpResDATA;
            _piDataLength = piDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwReceiveDataBCC(
                                                    _iPortNo,
                                                    _iACKResponse,
                                                    _lTimeOut,
                                                    _iDSRCheck,
                                                    ref _bpResDATA,
                                                    ref _piDataLength);
        }
    }
    #endregion

    #region (14) HF帯RFID電文ﾃﾞｰﾀ受信関数呼出しクラス
    /// <summary>
    /// (14) HF帯RFID電文ﾃﾞｰﾀ受信関数呼出しクラス
    /// </summary>
    class CrwReceiveDataBCCHFFunction : CrwFunction
    {
        #region HF電文データ受信関数 crwReceiveDataBCCHF()
        /// <summary>
        /// HF電文データ受信関数 crwReceiveDataBCCHF()
        /// </summary>
        /// <param name="iPortNo">レスポンスを受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iACKJudge">1レスポンス目の電文で２レスポンス目を返却するかどうかを判断させるフラグ（0以外:する / 0:しないの場合は1レスポンスのみの返却）</param>
        /// <param name="lTimeOut">レスポンス受信待ちタイマー（単位:msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0以外:する / 0:しない）</param>
        /// <param name="bpResReserveF">1レスポンス目のReserve部分を返却するエリアのポインタ(1バイト)</param>
        /// <param name="bpResRespF">1レスポンス目のコード部分を返却するエリアのポインタ(2バイト)</param>
        /// <param name="bpResStatF">1レスポンス目ステータス部分を返却するエリアのポインタ(2バイト)</param>
        /// <param name="bpResReserveS">２レスポンス目のReserve部分を返却するエリアのポインタ(1バイト)</param>
        /// <param name="bpResRespS">２レスポンス目のコード部分を返却するエリアのポインタ(2バイト)</param>
        /// <param name="bpResStatS">２レスポンス目のステータス部分を返却するエリアのポインタ(2バイト)</param>
        /// <param name="bpResDATA">１又は２レスポンス目のデータ部分を返却するエリアのポインタ</param>
        /// <param name="piDataLength">１又は２レスポンス目のデータ部分の長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwReceiveDataBCCHF(
                                        short iPortNo,
                                        short iACKJudge,
                                        uint lTimeOut,
                                        short iDSRCheck,
                                        ref System.IntPtr bpResReserveF,
                                        ref System.IntPtr bpResRespF,
                                        ref System.IntPtr bpResStatF,
                                        ref System.IntPtr bpResReserveS,
                                        ref System.IntPtr bpResRespS,
                                        ref System.IntPtr bpResStatS,
                                        ref System.IntPtr bpResDATA,
                                        ref short piDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iACKJudge;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public System.IntPtr _bpResReserveF;
        public System.IntPtr _bpResRespF;
        public System.IntPtr _bpResStatF;
        public System.IntPtr _bpResReserveS;
        public System.IntPtr _bpResRespS;
        public System.IntPtr _bpResStatS;
        public System.IntPtr _bpResDATA;
        public short _piDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iACKJudge"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="bpResReserveF"></param>
        /// <param name="bpResRespF"></param>
        /// <param name="bpResStatF"></param>
        /// <param name="bpResReserveS"></param>
        /// <param name="bpResRespS"></param>
        /// <param name="bpResStatS"></param>
        /// <param name="bpResDATA"></param>
        /// <param name="piDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwReceiveDataBCCHFFunction(
                                        short iPortNo,
                                        short iACKJudge,
                                        uint lTimeOut,
                                        short iDSRCheck,
                                        ref System.IntPtr bpResReserveF,
                                        ref System.IntPtr bpResRespF,
                                        ref System.IntPtr bpResStatF,
                                        ref System.IntPtr bpResReserveS,
                                        ref System.IntPtr bpResRespS,
                                        ref System.IntPtr bpResStatS,
                                        ref System.IntPtr bpResDATA,
                                        ref short piDataLength,
                                        CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iACKJudge = iACKJudge;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _bpResReserveF = bpResReserveF;
            _bpResRespF = bpResRespF;
            _bpResStatF = bpResStatF;
            _bpResReserveS = bpResReserveS;
            _bpResRespS = bpResRespS;
            _bpResStatS = bpResStatS;
            _bpResDATA = bpResDATA;
            _piDataLength = piDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwReceiveDataBCCHF(
                                        _iPortNo,
                                        _iACKJudge,
                                        _lTimeOut,
                                        _iDSRCheck,
                                        ref _bpResReserveF,
                                        ref _bpResRespF,
                                        ref _bpResStatF,
                                        ref _bpResReserveS,
                                        ref _bpResRespS,
                                        ref _bpResStatS,
                                        ref _bpResDATA,
                                        ref _piDataLength);
        }
    }
    #endregion

    #region (15) 受信バッファクリア関数呼出しクラス
    /// <summary>
    /// (15) 受信バッファクリア関数呼出しクラス
    /// </summary>
    class CrwComRXBufferClearFunction : CrwFunction
    {
        #region 受信バッファークリア関数 crwComRXBufferClear()
        /// <summary>
        /// 受信バッファークリア関数 crwComRXBufferClear()
        /// </summary>
        /// <param name="iPortNo">受信バッファーをクリアするシリアルポートの番号（例：COM1なら”1”）</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwComRXBufferClear(
                                         short iPortNo);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwComRXBufferClearFunction(
                                         short iPortNo,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwComRXBufferClear(_iPortNo);
        }
    }
    #endregion

    #region (16) UHF帯RFIDコマンド送受信関数呼出しクラス
    /// <summary>
    /// (16) UHF帯RFIDコマンド送受信関数呼出しクラス
    /// </summary>
    class CrwSendBinaryCommandRRFunction : CrwFunction
    {
        #region UHF帯RFIDコマンド送受信関数 crwSendBinaryCommandRR()
        /// <summary>
        /// UHF帯RFIDコマンド送受信関数 crwSendBinaryCommandRR()
        /// </summary>
        /// <param name="iPortNo">シリアルポートの番号</param>
        /// <param name="lTimeOut">ACK/NAK受信待ち、及びデータ送信時のCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iRetryCount">データ送信、NAK受信、及びLRC受信時のリトライ回数</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="lpSendData">送信するデータ</param>
        /// <param name="iSendDataLength">送信するデータの長さ</param>
        /// <param name="pTimeOut">データの受信待ちタイマー値（単位：msec）</param>
        /// <param name="lpReceiveData">受信データを返却するエリアのポインタ</param>
        /// <param name="iReceiveDataLength">受信データを返却するエリアのサイズ</param>
        /// <param name="piDataLengthReceived">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSendBinaryCommandRR(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpSendData,
                                         short iSendDataLength,
                                         uint pTimeOut,
                                         void* lpReceiveData,
                                         short iReceiveDataLength,
                                         ref short piDataLengthReceived);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iRetryCount;
        public short _iDSRCheck;
        public short _iCTSCheck;
        public /*unsafe void**/byte[] _lpSendData;
        public short _iSendDataLength;
        public uint _pTimeOut;
        public /*unsafe void**/byte[] _lpReceiveData;
        public short _iReceiveDataLength;
        public short _piDataLengthReceived;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="lpSendData"></param>
        /// <param name="iSendDataLength"></param>
        /// <param name="pTimeOut"></param>
        /// <param name="lpReceiveData"></param>
        /// <param name="iReceiveDataLength"></param>
        /// <param name="piDataLengthReceived"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwSendBinaryCommandRRFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpSendData,
                                         short iSendDataLength,
                                         uint pTimeOut,
                                         void* lpReceiveData,
                                         short iReceiveDataLength,
                                         ref short piDataLengthReceived,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iRetryCount = iRetryCount;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _lpSendData = /*lpSendData*/new byte[1024];
            _iSendDataLength = iSendDataLength;
            _pTimeOut = pTimeOut;
            _lpReceiveData = /*lpReceiveData*/new byte[1024];
            _iReceiveDataLength = iReceiveDataLength;
            _piDataLengthReceived = piDataLengthReceived;
            for (int i = 0; i < _lpSendData.GetLength(0); i++)
            {
                _lpSendData[i] = ((byte*)lpSendData)[i];
            }
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            byte* lpSendData = stackalloc byte[_lpSendData.GetLength(0)];
            for (int i = 0; i < _lpSendData.GetLength(0); i++)
            {
                lpSendData[i] = _lpSendData[i];
            }
            byte* lpReceiveData = stackalloc byte[_lpReceiveData.GetLength(0)];
            _ret = crwSendBinaryCommandRR(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iRetryCount,
                                         _iDSRCheck,
                                         _iCTSCheck,
                                        /*_lpSendData*/lpSendData,
                                         _iSendDataLength,
                                         _pTimeOut,
                                        /*_lpReceiveData*/lpReceiveData,
                                         _iReceiveDataLength,
                                         ref _piDataLengthReceived);
            for (int j = 0; j < _lpReceiveData.GetLength(0); j++)
            {
                _lpReceiveData[j] = lpReceiveData[j];
            }
        }
    }
    #endregion

    #region (17) UHF帯RFIDコマンド送信関数呼出しクラス
    /// <summary>
    /// (17) UHF帯RFIDコマンド送信関数呼出しクラス
    /// </summary>
    class CrwSendBinaryDataFunction : CrwFunction
    {
        #region UHF帯RFIDコマンド送信関数 crwSendBinaryData()
        /// <summary>
        /// UHF帯RFIDコマンド送信関数 crwSendBinaryData() 
        /// </summary>
        /// <param name="iPortNo">シリアルポートの番号</param>
        /// <param name="lTimeOut">ACK/NAK受信待ち、及びデータ送信時のCTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iRetryCount">データ送信、NAK受信、及びLRC受信時のリトライ回数</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSendBinaryData(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iRetryCount;
        public short _iDSRCheck;
        public short _iCTSCheck;
        public /*unsafe void**/byte[] _lpData;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwSendBinaryDataFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iRetryCount = iRetryCount;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _lpData = /*lpData*/new byte[2048];
            _iDataLength = iDataLength;
            for (int i = 0; i < _lpData.GetLength(0); i++)
            {
                _lpData[i] = ((byte*)lpData)[i];
            }
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            byte* lpData = stackalloc byte[_lpData.GetLength(0)];
            for (int i = 0; i < _lpData.GetLength(0); i++)
            {
                lpData[i] = _lpData[i];
            }
            _ret = crwSendBinaryData(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iRetryCount,
                                         _iDSRCheck,
                                         _iCTSCheck,
                                         /*_lpData*/lpData,
                                         _iDataLength);
        }
    }
    #endregion

    #region (18) UHF帯RFIDコマンド受信関数呼出しクラス
    /// <summary>
    /// (18) UHF帯RFIDコマンド受信関数呼出しクラス
    /// </summary>
    class CrwReceiveBinaryReportFunction : CrwFunction
    {
        #region UHF帯RFIDコマンド受信関数 crwReceiveBinaryReport()
        /// <summary>
        /// UHF帯RFIDコマンド受信関数 crwReceiveBinaryReport() 
        /// </summary>
        /// <param name="iPortNo">シリアルポートの番号</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値（単位：msec）</param>
        /// <param name="iRetryCount">LRC受信時のリトライ回数</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="lpData">受信データを返却するエリアのポインタ</param>
        /// <param name="iDataLength">受信データを返却するエリアのサイズ</param>
        /// <param name="piDataLength">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwReceiveBinaryReport(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength,
                                         ref short piDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iRetryCount;
        public short _iDSRCheck;
        public short _iCTSCheck;
        public /*unsafe void**/byte[] _lpData;
        public short _iDataLength;
        public short _piDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="piDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwReceiveBinaryReportFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength,
                                         ref short piDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iRetryCount = iRetryCount;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _lpData = /*lpData*/new byte[1024];
            _iDataLength = iDataLength;
            _piDataLength = piDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            byte* lpData = stackalloc byte[_lpData.GetLength(0)];
            _ret = crwReceiveBinaryReport(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iRetryCount,
                                         _iDSRCheck,
                                         _iCTSCheck,
                                         /*_lpData*/lpData,
                                         _iDataLength,
                                         ref _piDataLength);
            for (int i = 0; i < _lpData.GetLength(0); i++)
            {
                _lpData[i] = lpData[i];
            }
        }
    }
    #endregion

    #region (19) バイナリデータ送信関数ＱＲ呼出しクラス
    /// <summary>
    /// (19) バイナリデータ送信関数ＱＲ呼出しクラス
    /// </summary>
    class CrwSendBinaryDataQRFunction : CrwFunction
    {
        #region バイナリデータ送信関数 crwSendBinaryDataQR()
        /// <summary>
        /// バイナリデータ送信関数 crwSendBinaryDataQR() 
        /// </summary>
        /// <param name="iPortNo">シリアルポートの番号</param>
        /// <param name="lTimeOut">CTSオン待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="lpData">送信するデータ</param>
        /// <param name="iDataLength">送信するデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSendBinaryDataQR(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public short _iCTSCheck;
        public unsafe void* _lpData;
        public short _iDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwSendBinaryDataQRFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         void* lpData,
                                         short iDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _lpData = lpData;
            _iDataLength = iDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            _ret = crwSendBinaryDataQR(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         _iCTSCheck,
                                         _lpData,
                                         _iDataLength);
        }
    }
    #endregion

    #region (20) バイナリデータ受信関数呼出しクラス
    /// <summary>
    /// (20) バイナリデータ受信関数呼出しクラス
    /// </summary>
    class CrwReceiveBinaryDataFunction : CrwFunction
    {
        #region バイナリデータ受信関数 crwReceiveBinaryData()
        /// <summary>
        /// バイナリデータ受信関数 crwReceiveBinaryData() 
        /// </summary>
        /// <param name="iPortNo">受信するシリアルポートの番号</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値（単位：msec）</param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ（0：しない／0以外：する）</param>
        /// <param name="lpData">受信データを返却するエリアのポインタ</param>
        /// <param name="iDataLength">受信データを返却するエリアのサイズ</param>
        /// <param name="piDataLength">受信したデータの長さ</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwReceiveBinaryData(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         void* lpData,
                                         short iDataLength,
                                         ref short piDataLength);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public short _iDSRCheck;
        public byte[] _lpData;
        public short _iDataLength;
        public short _piDataLength;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="lpData"></param>
        /// <param name="iDataLength"></param>
        /// <param name="piDataLength"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwReceiveBinaryDataFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         short iDSRCheck,
                                         void* lpData,
                                         short iDataLength,
                                         ref short piDataLength,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _iDSRCheck = iDSRCheck;
            _lpData = /*lpData*/new byte[1024];
            _iDataLength = iDataLength;
            _piDataLength = piDataLength;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            byte* lpData = stackalloc byte[1024];
            _ret = crwReceiveBinaryData(
                                         _iPortNo,
                                         _lTimeOut,
                                         _iDSRCheck,
                                         lpData,
                                         _iDataLength,
                                         ref _piDataLength);
            for (int i = 0; i < _lpData.GetLength(0); i++)
            {
                _lpData[i] = ((byte*)lpData)[i];
            }
        }
    }
    #endregion

    #region (21) 通信中DSR/CTS信号監視設定関数呼出しクラス
    /// <summary>
    /// (21) 通信中DSR/CTS信号監視設定関数呼出しクラス
    /// </summary>
    class CrwComDSRCTSswFunction : CrwFunction
    {
        #region 通信中DSR/CTS信号監視設定関数 crwComDSRCTSsw()
        /// <summary>
        /// 通信中DSR/CTS信号監視設定関数 crwComDSRCTSsw()
        /// </summary>
        /// <param name="iPortNo">DSR・CTSフローを監視するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iDSRCheck">通信中のDSR信号線をチェックするかどうかのフラグ（0:しない／0以外:する）初期値：0以外:する</param>
        /// <param name="iCTSCheck">通信中のCTS信号線をチェックするかどうかのフラグ（0:しない／0以外:する）初期値：0以外:する</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwComDSRCTSsw(
                                         short iPortNo,
                                         short iDSRCheck,
                                         short iCTSCheck);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iDSRCheck;
        public short _iCTSCheck;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwComDSRCTSswFunction(
                                         short iPortNo,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwComDSRCTSsw(
                                         _iPortNo,
                                         _iDSRCheck,
                                         _iCTSCheck);
        }
    }
    #endregion

    #region (22) レスポンス受信イベントハンドラ登録関数呼出しクラス
    /// <summary>
    /// (22) レスポンス受信イベントハンドラ登録関数呼出しクラス
    /// </summary>
    class CrwSetResponseEventProcFunction : CrwFunction
    {
        #region レスポンス受信イベントハンドラ登録関数 crwSetResponseEventProc()
        /// <summary>
        /// レスポンス受信イベントハンドラ登録関数 crwSetResponseEventProc() 
        /// </summary>
        /// <param name="iPortNo">受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iRecMode">ビジーステータス受信時にポーリング再送を行うかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値(単位:msec) （0 msec指定で無限待ち）</param>
        /// <param name="pResponseEventProc">crwResponseEventProc関数のポインタ(NULLを指定:登録解除)</param>
        /// <param name="lParam">アプリケーションによって定義された値</param>
        /// <param name="lEvent">コールバック関数で受け取るイベントの論理和</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSetResponseEventProc(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         CallBack pResponseEventProc,
                                         int lParam,
                                         uint lEvent);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iRecMode;
        public uint _lTimeOut;
        public CallBack _pResponseEventProc;
        public int _lParam;
        public uint _lEvent;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRecMode"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="pResponseEventProc"></param>
        /// <param name="lParam"></param>
        /// <param name="lEvent"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetResponseEventProcFunction(
                                         short iPortNo,
                                         short iRecMode,
                                         uint lTimeOut,
                                         CallBack pResponseEventProc,
                                         int lParam,
                                         uint lEvent,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRecMode = iRecMode;
            _lTimeOut = lTimeOut;
            _pResponseEventProc = pResponseEventProc;
            _lParam = lParam;
            _lEvent = lEvent;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetResponseEventProc(
                                         _iPortNo,
                                         _iRecMode,
                                         _lTimeOut,
                                         _pResponseEventProc,
                                         _lParam,
                                         _lEvent);
        }
    }
    #endregion

    #region (23) バイナリデータ受信イベントハンドラ登録関数呼出しクラス
    /// <summary>
    /// (23) バイナリデータ受信イベントハンドラ登録関数呼出しクラス
    /// </summary>
    class CrwSetReceiveBinaryEventProcFunction : CrwFunction
    {
        #region バイナリデータ受信イベントハンドラ登録関数 crwSetReceiveBinaryEventProc()
        /// <summary>
        /// バイナリデータ受信イベントハンドラ登録関数 crwSetReceiveBinaryEventProc() 
        /// </summary>
        /// <param name="iPortNo">受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値(単位:msec)（0 msec指定で無限待ち）</param>
        /// <param name="pReceiveBinaryEventProc">crwReceiveBinaryEventProc関数のポインタ(NULLを指定:登録解除)</param>
        /// <param name="pParam">アプリケーションによって定義された値</param>
        /// <param name="lEvent">コールバック関数で受け取るイベントの論理和</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSetReceiveBinaryEventProc(
                                         short iPortNo,
                                         uint lTimeOut,
                                         CallBack2 pReceiveBinaryEventProc,
                                         int pParam,
                                         uint lEvent);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public uint _lTimeOut;
        public CallBack2 _pReceiveBinaryEventProc;
        public int _pParam;
        public uint _lEvent;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="pReceiveBinaryEventProc"></param>
        /// <param name="pParam"></param>
        /// <param name="lEvent"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetReceiveBinaryEventProcFunction(
                                         short iPortNo,
                                         uint lTimeOut,
                                         CallBack2 pReceiveBinaryEventProc,
                                         int pParam,
                                         uint lEvent,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _lTimeOut = lTimeOut;
            _pReceiveBinaryEventProc = pReceiveBinaryEventProc;
            _pParam = pParam;
            _lEvent = lEvent;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetReceiveBinaryEventProc(
                                         _iPortNo,
                                         _lTimeOut,
                                         _pReceiveBinaryEventProc,
                                         _pParam,
                                         _lEvent);
        }
    }
    #endregion

    #region (24) 電文データ受信イベントハンドラ登録関数呼出しクラス
    /// <summary>
    /// (24) 電文データ受信イベントハンドラ登録関数呼出しクラス
    /// </summary>
    class CrwSetReceiveDataBCCEventProcFunction : CrwFunction
    {
        #region 電文データ受信イベントハンドラ登録関数 crwSetReceiveDataBCCEventProc()
        /// <summary>
        /// レ電文データ受信イベントハンドラ登録関数 crwSetReceiveDataBCCEventProc() 
        /// </summary>
        /// <param name="iPortNo">受信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iACKResponse">ACK/NAKを返却するかどうかのフラグ（0以外：する / 0：しない）</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値(単位:msec) （0 msec指定で無限待ち）</param>
        /// <param name="pReceiveBCCEventProc">crwSetReceiveDataBCCEventProc関数のポインタ(NULLを指定:登録解除)</param>
        /// <param name="lParam">アプリケーションによって定義された値</param>
        /// <param name="lEvent">コールバック関数で受け取るイベントの論理和</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSetReceiveDataBCCEventProc(
                                         short iPortNo,
                                         short iACKResponse,
                                         uint lTimeOut,
                                         CallBack pReceiveBCCEventProc,
                                         int lParam,
                                         uint lEvent);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iACKResponse;
        public uint _lTimeOut;
        public CallBack _pReceiveBCCEventProc;
        public int _lParam;
        public uint _lEvent;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iACKResponse"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="pReceiveBCCEventProc"></param>
        /// <param name="lParam"></param>
        /// <param name="lEvent"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetReceiveDataBCCEventProcFunction(
                                         short iPortNo,
                                         short iACKResponse,
                                         uint lTimeOut,
                                         CallBack pReceiveBCCEventProc,
                                         int lParam,
                                         uint lEvent,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iACKResponse = iACKResponse;
            _lTimeOut = lTimeOut;
            _pReceiveBCCEventProc = pReceiveBCCEventProc;
            _lParam = lParam;
            _lEvent = lEvent;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetReceiveDataBCCEventProc(
                                         _iPortNo,
                                         _iACKResponse,
                                         _lTimeOut,
                                         _pReceiveBCCEventProc,
                                         _lParam,
                                         _lEvent);
        }
    }
    #endregion

    #region (25) 通信イベントハンドラ登録関数呼出しクラス
    /// <summary>
    /// (25) 通信イベントハンドラ登録関数呼出しクラス
    /// </summary>
    class CrwSetCommEventProcFunction : CrwFunction
    {
        #region 通信イベントハンドラ登録関数 crwSetCommEventProc()
        /// <summary>
        /// 通信イベントハンドラ登録関数 crwSetCommEventProc()
        /// </summary>
        /// <param name="iPortNo">監視するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="pCommEventProc">crwCommEventProc関数のポインタ(NULLを指定:登録解除)</param>
        /// <param name="pParam">アプリケーションによって定義された値</param>
        /// <param name="lEvent">コールバック関数で受け取るイベントの論理和</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        public static extern short crwSetCommEventProc(
                                         short iPortNo,
                                         CallBack3 pCommEventProc,
                                         int pParam,
                                         uint lEvent);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public CallBack3 _pCommEventProc;
        public int _pParam;
        public uint _lEvent;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="pCommEventProc"></param>
        /// <param name="pParam"></param>
        /// <param name="lEvent"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetCommEventProcFunction(
                                         short iPortNo,
                                         CallBack3 pCommEventProc,
                                         int pParam,
                                         uint lEvent,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _pCommEventProc = pCommEventProc;
            _pParam = pParam;
            _lEvent = lEvent;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetCommEventProc(
                                         _iPortNo,
                                         _pCommEventProc,
                                         _pParam,
                                         _lEvent);
        }
    }
    #endregion

    #region (26) AutoHandShake設定関数呼出しクラス
    /// <summary>
    /// (26) AutoHandShake設定関数関数呼出しクラス
    /// </summary>
    class CrwSetHandshakeFunction : CrwFunction
    {
        #region AutoHandShake設定関数 crwSetHandshake()
        /// <summary>
        /// AutoHandShake設定関数関数 crwSetHandshake()
        /// </summary>
        /// <param name="iPortNo">Handshakeを設定するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iHandshake">HandShakeを設定するかどうかのフラグ（0:しない／0以外:する）初期値：0以外:する</param>
         /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwSetHandshake(
                                         short iPortNo,
                                         short iHandshake);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iHandshake;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iHandshake"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetHandshakeFunction(
                                         short iPortNo,
                                         short iHandshake,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iHandshake = iHandshake;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetHandshake(
                                         _iPortNo,
                                         _iHandshake);
        }
    }
    #endregion

    #region (27) AutoHandShake状態取得関数呼出しクラス

    /// <summary>
    /// (27) AutoHandShake状態取得関数呼出しクラス
    /// </summary>
    // CrwFunctionクラスを継承してクラスを作成
    class CrwGetHandshakeStatusFunction : CrwFunction
    {
        #region AutoHandShake状態取得関数関数 crwGetHandshakeStatus()
        /// <summary>
        /// AutoHandShake状態取得関数関数 crwGetHandshakeStatus()
        /// </summary>
        /// <param name="iPortNo">各信号状態を取得するシリアルポートの番号(例：COM1なら"1")</param>
        /// <param name="piAutoHandshake">Handshake状態を取得するためのエリアのポインタ(1バイト)(0:OFF/1:ON)</param>
        /// <returns>整数型(short)</returns>

        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwGetHandshakeStatus(
                                         short iPortNo,
                                         ref short piAutoHandshake);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _piAutoHandshake;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="piAutoHandshake"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwGetHandshakeStatusFunction(
                                         short iPortNo,
                                         ref short piAutoHandshake,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _piAutoHandshake = piAutoHandshake;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            _ret = crwGetHandshakeStatus(
                                         _iPortNo,
                                         ref _piAutoHandshake);
        }
    }

    #endregion

    #region (28) 信号線状態取得関数呼び出しクラス
    
    /// <summary>
    /// (28) 信号線状態取得関数呼び出しクラス
    /// </summary>
    class CrwGetLineStatusFunction : CrwFunction
    {
        #region 信号線状態取得関数 crwGetLineStatus()
        /// <summary>
        /// 信号線状態取得関数 crwGetLineStatus()
        /// </summary>
        /// <param name="iPortNo">各信号状態を取得するシリアルポートの番号(例:COM1なら"1")</param>
        /// <param name="iCTSLine">CTS信号の状態を取得するためのエリアのポインタ(1バイト)(0:OFF/1:ON)</param>
        /// <param name="iDSRLine">DSR信号の状態を取得するためのエリアのポインタ(1バイト)(0:OFF/1:ON)</param>        
        /// <param name="iRLSDLine">RLSD信号の状態を取得するためのエリアのポインタ(1バイト)(0:OFF/1:ON)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        unsafe static extern short crwGetLineStatus(
                                        short iPortNo,
                                        ref short piCTSLine,
                                        ref short piDSRLine,
                                        ref short piRLSDLine);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _piCTSLine;
        public short _piDSRLine;
        public short _piRLSDLine;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iHandshake"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public unsafe CrwGetLineStatusFunction(
                                        short iPortNo,
                                        ref short piCTSLine,
                                        ref short piDSRLine,
                                        ref short piRLSDLine,
                                        CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _piCTSLine = piCTSLine;
            _piDSRLine = piDSRLine;
            _piRLSDLine = piRLSDLine;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected unsafe override void DoCall()
        {
            _ret = crwGetLineStatus(
                                         _iPortNo,
                                         ref _piCTSLine,
                                         ref _piDSRLine,
                                         ref _piRLSDLine);
        }
    }

    #endregion

    #region (29) 信号線制御関数呼び出しクラス
    /// <summary>
    /// (29) 信号線制御関数呼び出しクラス
    /// </summary>
    class CrwSetLineControlFunction : CrwFunction
    {
        #region 信号線制御関数呼び出しクラス crwSetLineControl()
        /// <summary>
        /// 信号線制御関数呼び出しクラス crwSetLineControl()
        /// </summary>
        /// <param name="iPortNo">各信号状態を取得するシリアルポートの番号（例：COM1なら"1"）</param>
        /// <param name="iDTRLine">DTR信号を制御するかどうかのフラグ(1バイト) (0:OFF/0以外:ON)</param>
        /// <param name="iRTSLine">RTS信号を制御するかどうかのフラグ(1バイト) (0:OFF/0以外:ON)</param>
        /// <param name="iBreakLine">Break信号を制御するかどうかのフラグ(1バイト) (0:OFF/0以外:ON)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        public static extern short crwSetLineControl(
                                         short iPortNo,
                                         short iDTRLine,
                                         short iRTSLine,
                                         short iBreakLine);
        #endregion

        #region メンバフィールド定義
        public short _ret;
        public short _iPortNo;
        public short _iDTRLine;
        public short _iRTSLine;
        public short _iBreakLine;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iDTRLine"></param>
        /// <param name="iRTSLine"></param>
        /// <param name="iBreakLine"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwSetLineControlFunction(
                                         short iPortNo,
                                         short iDTRLine,
                                         short iRTSLine,
                                         short iBreakLine,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iDTRLine = iDTRLine;
            _iRTSLine = iRTSLine;
            _iBreakLine = iBreakLine;
        }

        /// <summary>
        /// CRW関数呼出し


        /// </summary>
        protected override void DoCall()
        {
            _ret = crwSetLineControl(
                                         _iPortNo,
                                         _iDTRLine,
                                         _iRTSLine,
                                         _iBreakLine);
        }
    }
    #endregion
  
    #region (20) PI Image Data 作成＆送信関数 crwPIDataSend()呼出しクラス
    /// <summary>
    /// (20) PI Image Data 作成＆送信関数 crwPIDataSend()呼出しクラス
    /// </summary>
    class CrwPIDataSendFunction : CrwFunction
    {
        #region コマンド送受信関数 crwPIDataSend()
        /// <summary>
        /// コマンド送受信関数 crwPIDataSend() 
        /// </summary>
        /// <param name="iPortNo">送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iRetryCount">データ送信、NAK 受信、及び SUNエラー受信時のリトライ回数 </param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ　（0:しない／0以外:する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0 以外：する）</param>
        /// <param name="iWidth">印字エリアの横幅（固定値：320）</param>
        /// <param name="iHeith">印字エリアの縦幅（機器の仕様書参考）例：460</param>
        /// <param name="iStartX">印字開始位置(X座標) 例：100</param>
        /// <param name="iStartY">印字開始位置(Y座標) 例：100</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値(単位:msec)</param>
        /// <param name="lComBufferSize">COMポート通信バッファーサイズ</param>
        /// <param name="strPathFileName">イメージデータファイル名[ BMP, JPG, GIF ] 形式（フルパス指定可能）</param>
        /// <param name="bpERRSEN">レスポンスのエラーコード(=ERR$)・レディステータスのセンサー情報(=SEN$)を返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分(=DATA$)を返却するエリアのポインタ(センサー情報以外の追加情報がある場合は、以降の情報は“bpResDATA”に格納される)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwPIDataSend(
                                         short iPortNo,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         short iWidth,
                                         short iHeith,
                                         short iStartX,
                                         short iStartY,
                                         uint lTimeOut,
                                         uint lComBufferSize,
                                         System.String strPathFileName,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN);

        #endregion

        #region メンバフィールド定義
            public short _ret;
            public short _iPortNo;
            public short _iRetryCount;
            public short _iDSRCheck;
            public short _iCTSCheck;
            public short _iWidth;
            public short _iHeith;
            public short _iStartX;
            public short _iStartY;
            public uint _lTimeOut;
            public uint _lComBufferSize;
            public System.String _strPathFileName;
            public System.IntPtr _bpCOMSTS;
            public System.IntPtr _bpERRSEN;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="iWidth"></param>
        /// <param name="iHeith"></param>
        /// <param name="iStartX"></param>
        /// <param name="iStartY"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="lComBufferSize"></param>
        /// <param name="strPathFileName"></param>
        /// <param name="bpCOMSTS"></param>
        /// <param name="bpERRSEN"></param>
        /// <param name="crwFunctionResultHandler"></param>
        public CrwPIDataSendFunction(
                                        short iPortNo,
                                        short iRetryCount,
                                        short iDSRCheck,
                                        short iCTSCheck,
                                        short iWidth,
                                        short iHeith,
                                        short iStartX,
                                        short iStartY,
                                        uint lTimeOut,
                                        uint lComBufferSize,
                                        System.String strPathFileName,
                                        ref System.IntPtr bpCOMSTS,
                                        ref System.IntPtr bpERRSEN,
                                     CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRetryCount = iRetryCount;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _iWidth = iWidth;
            _iHeith = iHeith;
            _iStartX = iStartX;
            _iStartY = iStartY;
            _lTimeOut = lTimeOut;
            _lComBufferSize = lComBufferSize;
            _strPathFileName = strPathFileName;
            _bpCOMSTS = bpCOMSTS;
            _bpERRSEN = bpERRSEN;

        }

        /// <summary>
        /// CRW関数呼出し
        /// </summary>
        protected override void DoCall()
        {
            _ret = crwPIDataSend(
                                 _iPortNo,
                                 _iRetryCount,
                                 _iDSRCheck,
                                 _iCTSCheck,
                                 _iWidth,
                                 _iHeith,
                                 _iStartX,
                                 _iStartY,
                                 _lTimeOut,
                                 _lComBufferSize,
                                 _strPathFileName,
                                 ref _bpCOMSTS,
                                 ref _bpERRSEN);
        }
    }
    #endregion

    #region (21) IB Image Data 作成＆送信関数 crwIBBinarySend()呼出しクラス
    /// <summary>
    /// (21) IB Image Data 作成＆送信関数 crwIBBinarySend()呼出しクラス
    /// </summary>
    class CrwIBBinarySendFunction : CrwFunction
    {
        #region IB Image Data 作成＆送信関数 crwIBBinarySend()
        /// <summary>
        /// IB Image Data 作成＆送信関数 crwIBBinarySend() 
        /// </summary>
        /// <param name="iPortNo">送信するシリアルポートの番号（例：COM1なら”1”）</param>
        /// <param name="iRetryCount">データ送信、NAK 受信、及び SUNエラー受信時のリトライ回数 </param>
        /// <param name="iDSRCheck">DSR信号線をチェックするかどうかのフラグ　（0:しない／0以外:する）</param>
        /// <param name="iCTSCheck">CTS信号線をチェックするかどうかのフラグ（0：しない／0 以外：する）</param>
        /// <param name="iWidth">印字エリアの横幅（固定値：320）</param>
        /// <param name="iHeith">印字エリアの縦幅（機器の仕様書参考）例：460</param>
        /// <param name="iStartY">印字開始位置(Y座標) 例：100</param>
        /// <param name="lTimeOut">データの受信待ちタイマー値(単位:msec)</param>
        /// <param name="lComBufferSize">COMポート通信バッファーサイズ</param>
        /// <param name="strPathFileName">イメージデータファイル名[ BMP, JPG, GIF ] 形式（フルパス指定可能）</param>
        /// <param name="bpERRSEN">レスポンスのエラーコード(=ERR$)・レディステータスのセンサー情報(=SEN$)を返却するエリアのポインタ（2バイト）</param>
        /// <param name="bpResDATA">レスポンスのデータ部分(=DATA$)を返却するエリアのポインタ(センサー情報以外の追加情報がある場合は、以降の情報は“bpResDATA”に格納される)</param>
        /// <returns>整数型(short)</returns>
        [DllImport("PCardRW32.dll")]
        static extern short crwIBBinarySend(
                                         short iPortNo,
                                         short iRetryCount,
                                         short iDSRCheck,
                                         short iCTSCheck,
                                         short iWidth,
                                         short iHeith,
                                         short iStartY,
                                         uint lTimeOut,
                                         uint lComBufferSize,
                                         System.String strPathFileName,
                                         ref System.IntPtr bpCOMSTS,
                                         ref System.IntPtr bpERRSEN);

        #endregion

        #region メンバフィールド定義
            public short _ret;
            public short _iPortNo;
            public short _iRetryCount;
            public short _iDSRCheck;
            public short _iCTSCheck;
            public short _iWidth;
            public short _iHeith;
            public short _iStartY;
            public uint _lTimeOut;
            public uint _lComBufferSize;
            public System.String _strPathFileName;
            public System.IntPtr _bpCOMSTS;
            public System.IntPtr _bpERRSEN;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="iPortNo"></param>
        /// <param name="iRetryCount"></param>
        /// <param name="iDSRCheck"></param>
        /// <param name="iCTSCheck"></param>
        /// <param name="iWidth"></param>
        /// <param name="iHeith"></param>
        /// <param name="iStartY"></param>
        /// <param name="lTimeOut"></param>
        /// <param name="lComBufferSize"></param>
        /// <param name="strPathFileName"></param>
        /// <param name="bpCOMSTS"></param>
        /// <param name="bpERRSEN"></param>
        /// <param name="crwFunctionResultHandler"></param>
            public CrwIBBinarySendFunction(
                                            short iPortNo,
                                            short iRetryCount,
                                            short iDSRCheck,
                                            short iCTSCheck,
                                            short iWidth,
                                            short iHeith,
                                            short iStartY,
                                            uint lTimeOut,
                                            uint lComBufferSize,
                                            System.String strPathFileName,
                                            ref System.IntPtr bpCOMSTS,
                                            ref System.IntPtr bpERRSEN,
                                         CrwFunctionResultHandler crwFunctionResultHandler)
            : base(crwFunctionResultHandler)
        {
            _ret = 0;
            _iPortNo = iPortNo;
            _iRetryCount = iRetryCount;
            _iDSRCheck = iDSRCheck;
            _iCTSCheck = iCTSCheck;
            _iWidth = iWidth;
            _iHeith = iHeith;
            _iStartY = iStartY;
            _lTimeOut = lTimeOut;
            _lComBufferSize = lComBufferSize;
            _strPathFileName = strPathFileName;
            _bpCOMSTS = bpCOMSTS;
            _bpERRSEN = bpERRSEN;

        }

        /// <summary>
        /// CRW関数呼出し
        /// </summary>
        protected override void DoCall()
        {
            _ret = crwIBBinarySend(
                                     _iPortNo,
                                     _iRetryCount,
                                     _iDSRCheck,
                                     _iCTSCheck,
                                     _iWidth,
                                     _iHeith,
                                     _iStartY,
                                     _lTimeOut,
                                     _lComBufferSize,
                                     _strPathFileName,
                                     ref _bpCOMSTS,
                                     ref _bpERRSEN);
        }
    }
    #endregion

}